#|______________________________________________________________________________
 |
 | plots001.lsp
 |
 | Contains the following constructor functions for the new ViSta Plots System:
 |
 |   dot-plot             1 dimensional point plot
 |   scatter-plot         2 dimensional point plot
 |   line-plot            2 dimensional connected point plot
 |   all-scatter-plots    all pairs of 2 dimensional point plots
 |   spinning-points      3 dimensional spinnable point plot
 |   orbiting-points      h dimensional orbiting points plot
 |______________________________________________________________________________
 |#


(defun scatter-plot 
  (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) (connect nil)
             (location '(50 50)) (size '(320 320))
             (title "Scatter Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"SCATTER-PLOT makes a scatterplot of the first two variables in the data. Scatterplots can be used to display the relationship between two variables. The two variables are represented by the X and Y axes. The observations are represented as points in the plot.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"ScatterPlot\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."

  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types ))
         (data-object (fourth graph-data))
         (linkable t)
         (graph))
    (unless variable-labels (setf variable-labels (second graph-data)))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    (unless legend2 (setf legend2 title))
    (setf graph (send vista-scatterplot-proto :new  graph-data                        
                      :variable-labels variable-labels :point-labels point-labels
                      :connect connect
                      :show nil                        :top-most top-most
                      :container container             :pop-out pop-out
                      :location location               :size size 
                      :menu menu                       :title title 
                      :legend1 legend1                 :legend2 legend2
                      :go-away go-away                 :content-only content-only))
    
    (if actcon 
        (enable-container actcon)
        (disable-container))
    (when graph
          (send graph :title (strcat title " [" (send graph :legend1) "]"))
          (send graph :make-two-plot-menus
                "Scatter"
                :hotspot-items '(help dash new-x new-y dash link dash
                                 show-plots hide-plots close-plots dash 
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          (send graph :after-new-plot 
                pop-out top-most show size container linkable linked
                data-object content-only)
          (when linkable (send graph :linked linked))
          )
    graph))




(defun dot-plot 
  (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) (connect nil)
             (location '(50 50)) (size '(320 320))
             (title "Dot Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"DOT-PLOT makes a scatterplot of the first variable versus the observation sequence number. Dotplots can be used to determine if there is a relationship between a variable and the order in which it appears in the data. The two variables are represented by the X and Y axes. The observations are represented as points in the plot.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"DotPlot\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."
  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types ))
         (data-object (fourth graph-data))
         (linkable t)
         (graph))
    (unless variable-labels (setf variable-labels (second graph-data)))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    
    (unless legend2 (setf legend2 title))

    (setf graph (send vista-dotplot-proto :new graph-data 
                      :variable-labels variable-labels
                      :point-labels point-labels
                      :show nil
                      :location location
                      :size size
                      :title title
                      :connect-points connect
                      :container container
                      :new-x t
                      :legend1 legend1
                      :legend2 legend2
                      :go-away go-away
                      :content-only content-only))
    
    (if actcon 
        (enable-container actcon)
        (disable-container))
    (when graph
          (send graph :legend1 
                (if data-object (send data-object :name)
                    "Unnamed Data"))
          (send graph :title (strcat title " [" (send graph :legend1) "]"))
          
          (send graph :after-new-plot 
                pop-out top-most show size container linkable linked
                data-object content-only)
          (when linkable (send graph :linked linked))
          
          (send graph :make-two-plot-menus
                "DotPlot"
                :hotspot-items '(help dash new-x dash link dash
                                 show-plots hide-plots close-plots dash 
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          )
    graph))





(defproto vista-dotplot-proto 
  '(spreadplot-supervisor showing legend1 legend2 legend3L legend3R container)
  '() scatterplot-proto)

(defmeth vista-dotplot-proto :isnew (&optional data 
        &key variable-labels point-labels connect-points
             (container nil) (show t) (top-most t) (pop-out nil)
             (location '(50 50)) (size '(300 300)) (new-x t)
             (title "Time Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             (go-away t) (content-only nil)) 
  (let* (
         )
    (setf variable-labels (combine (first variable-labels) 
                                   "Observation Number" (rest variable-labels)))
    (setf data 
          (if (rest data)
              (append (append (list (first data))
                              (list (coerce (1+ (iseq (length (first data)))) 
                                            'vector)))
                      (rest data))
              (append (list (first data))
                      (list (coerce (1+ (iseq (length (first data))))
                                    'vector)))))
    (call-next-method (length data)
                      :variable-labels variable-labels
                      :point-labels point-labels
                      :show nil
                      :location location
                      :size size
                      :title title
                      )
    (send self :use-color t)
    (send self :add-points data)
    (when connect-points (send self :add-lines data :color 'blue))
    (send self :pop-out-on pop-out)
    (send self :top-most-on top-most)
    (send self :plot-buttons :new-x new-x :new-y nil :free nil :density nil)
    (send self :point-color (iseq (send self :num-points)) 'blue)
    (send self :mouse-mode 'brushing)
    (send self :x-axis t t (third (send self :x-axis)))
    (send self :y-axis t t (third (send self :y-axis)))
    (send self :point-label (iseq (send self :num-points)) point-labels)
    (send self :showing-labels t)
    (send self :legend1 legend1)
    (send self :legend2 legend2)
    (send self :adjust-to-data)
    (send self :container container)
    (when show (send self :show-window))
    (setf *graph* self)
    self))


(defmeth vista-dotplot-proto :legend1 (&optional (string nil set))
  (if set (setf (slot-value 'legend1) string))
  (slot-value 'legend1))

(defmeth vista-dotplot-proto :legend2 (&optional (string nil set))
  (if set (setf (slot-value 'legend2) string))
  (slot-value 'legend2))

(defmeth vista-dotplot-proto :legend3L (&optional (string nil set))
  (if set (setf (slot-value 'legend3L) string))
  (slot-value 'legend3L))

(defmeth vista-dotplot-proto :legend3R (&optional (string nil set))
  (if set (setf (slot-value 'legend3R) string))
  (slot-value 'legend3R))

(defmeth vista-dotplot-proto :container (&optional (object nil set))
  (if set (setf (slot-value 'container) object))
  (slot-value 'container))

(defmeth vista-dotplot-proto :redraw-content ()
  (call-next-method)
  (send self :draw-color 'black)
  (send self :draw-legends)
  (send self :add-grid))


(defun line-plot 
  (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) 
             (connect t) (connect-points t)
             (location '(50 50)) (size '(320 320))
             (title "Line Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             ;following dont work
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"LINE-PLOT makes a connected scatterplot of the first variable versus the observation sequence number. Line plots can be used to determine if there is a relationship between a variable and the order in which it appears in the data. The two variables are represented by the X and Y axes. The observations are represented as connected points in the plot.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"DotPlot\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."
  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types))
         (data-object (fourth graph-data))
         (linkable t)
         (nice)
         (graph))
    (when (not (eql connect connect-points))
          (error-message "cannot use both connect and connect-points"))
    (unless variable-labels 
            (setf variable-labels (second graph-data)))
    (setf variable-labels (combine "Observations" variable-labels))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    (setf graph-data 
          (append (list (coerce (1+ (iseq (length (first graph-data)))) 'vector)) 
                  graph-data))
    (setf nice (get-nice-range 0 (1+ (length (first graph-data))) 6))
    (unless legend2 (setf legend2 title))
    (setf graph (send vista-scatterplot-proto          :new  graph-data
                      :variable-labels variable-labels :point-labels point-labels
                      :connect-points connect-points   :lines nil
                      :new-x nil                       :new-y t
                      :show nil                        :top-most top-most
                      :container container             :pop-out pop-out
                      :location location               :size size 
                      :menu menu                       :title title 
                      :legend1 legend1                 :legend2 legend2
                      :go-away go-away                 :content-only content-only))
    (if actcon 
        (enable-container actcon)
        (disable-container))
    (send graph :range 0 (first nice) (second nice))
    (send graph :x-axis t t (third nice))
    (when graph
          (send graph :legend1 
                (if data-object (send data-object :name)
                    "Unnamed Data"))
          (send graph :title (strcat title " [" (send graph :legend1) "]"))
          (send graph :make-two-plot-menus
                "LinePlot"
                :hotspot-items '(help dash new-x new-y dash link dash
                                 show-plots hide-plots close-plots dash 
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          (send graph :after-new-plot 
                pop-out top-most show size container linkable linked 
                data-object content-only)
          (when linkable (send graph :linked linked))
          )
    graph))


(defun scatter-matrix 
  (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) (connect nil)
             (location '(50 50)) (size '(320 320))
             (title "Scatter Matrix") 
             (legend1 "Unnamed Data") (legend2 nil)
             ;following dont work
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"SCATTER-MATRIX makes a matrix (table) of scatterplots of all pairs of numeric variables in the data. The first nine or fewer variables are used. The plot facilitates comparison the relationships shown between each scatterplot's pair of variables.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"Scatter Matrix\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."
  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types ))
         (data-object (fourth graph-data))
         (linkable t)
         (graph))
    (unless variable-labels (setf variable-labels (second graph-data)))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    (unless legend2 (setf legend2 title))
    (setf graph 
          (scatmat graph-data                        
                              :variable-labels variable-labels 
                              :point-labels point-labels
                              :show nil                        
                              :top-most top-most
                              :container container           
                              :pop-out pop-out
                              :location location 
                              :size size 
                              :menu nil
                              :title title 
                              :legend1 legend1  
                              :legend2 legend2
                              :go-away go-away       
                              :content-only content-only))
    
    (if actcon 
        (enable-container actcon)
        (disable-container))
    (when graph
          (send graph :title (strcat title 
                                     " [" 
                                     (if data-object (send data-object :name)
                                         "Unnamed Data")
                                     "]"))
          (send graph :point-color (iseq (send graph :num-points)) 'blue)
          (send graph :make-two-plot-menus
                "AllScat"
                :hotspot-items '(help dash link dash
                                 show-plots hide-plots close-plots dash 
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          (send graph :after-new-plot 
                pop-out top-most show size container linkable linked
                data-object content-only)
          (when linkable (send graph :linked linked))
          )
    graph))



(defun spinning-points 
  (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) (connect nil)
             (location '(50 50)) (size '(320 320))
             (title "Spinning Points Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             ;following dont work
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"SPINNING-POINTS makes a three-dimensional spinnable scatterplot of the first three numeric variables in the data. Scatterplots can be used to display the relationship between variables. The variables are represented by the X, Y and Z axes. The observations are represented as points in the 3D space.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"Spinning Points Plot\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."
  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types))
         (data-object (fourth graph-data))
         (linkable t)
         (graph))
    (unless variable-labels (setf variable-labels (second graph-data)))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    (unless legend2 (setf legend2 title))
    (setf graph (send vista-spinplot-proto :new  graph-data                        
                      :variable-labels variable-labels :point-labels point-labels
                      :show nil                        :top-most top-most
                      :container container             :pop-out pop-out
                      :actcon actcon
                      :location location               :size size 
                      :menu menu                       :title title 
                      :legend1 legend1                 :legend2 legend2
                      :go-away go-away                 :content-only content-only))
    (when graph
          (send graph :title (strcat 
             title " [" (if data-object (send data-object :name) "Unnamed Data") "]"))
         ; (send spin-proto :menu-template (append (send graph-proto :menu-template)
         ;                               '(dash faster slower cuing axes)))
          (send graph :make-two-plot-menus
                "SpinPlot"
                :hotspot-items '(help dash new-x new-y new-z dash link dash
                                 faster slower cuing dash
                                 axes box dash
                                 show-plots hide-plots close-plots dash 
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          (send graph :after-new-plot 
                pop-out top-most show size container linkable linked
                data-object content-only)
          (when linkable (send graph :linked linked))
          (send graph :scaled-range 
                (iseq (send graph :num-variables)) -2 2))
    graph))


(defun orbiting-points
   (&optional data &key (ok-var-types '(numeric)) 
             variable-labels point-labels 
             (show t) (top-most t) (linked t) (connect nil)
             (location '(50 50)) (size '(320 320))
             (title "Orbiting Points Plot") 
             (legend1 "Unnamed Data") (legend2 nil)
             ;dont use menu-item and menu 
             ;following dont work
             (content-only nil)  (go-away t)
             ;do not use menu-item, menu, in, in?, in??, or in???
             ;all are reserved for system use 
             (menu-item nil) (menu t)
             (in nil in??) (in? nil in???))
"ORBITING-POINTS makes a six-dimensional spinnable scatterplot of the first six numeric variables in the data. Scatterplots can be used to display the relationship between variables. The variables are represented by the six axes. The observations are represented as points in the 6D space which are orbiting arround their centroid. Points which are close together in 6D space will have similar oribits.

ARGUMENTS:
&optional data &key (ok-var-types '(numeric)) variable-labels point-labels (show t) (top-most t) (linked t) (connect nil) (location '(50 50)) (size '(320 320)) (title \"Orbiting Points Plot\") (legend1 (send $ :name)) (legend2 (send self :title)) (content-only nil)

The optional DATA argument may be omitted or may be NIL, or it may be a data object, a list of numeric or string elements, a list of variable objects, a list of equal-length lists, a vector, a list of equal-length vectors, or a matrix. If omitted or NIL, DATA is assumed to be $, the current dataobject. In all cases, DATA is converted into a list of equal-length vectors. one vector for each of the OK-VAR-TYPES (see below) variables in the referenced data object."
  (let* ((actcon *active-container*)
         (in? (if in??? in? in??))
         (graph-frame (graph-frame :show nil))
         (pop-out (send graph-frame :seen-in in in?))
         (container (if (equal pop-out t) graph-frame pop-out))
         (graph-data (before-new-plot data ok-var-types ))
         (data-object (fourth graph-data))
         (linkable t)
         (graph))
    (unless variable-labels (setf variable-labels (second graph-data)))
    (unless point-labels (setf point-labels (third graph-data)))
    (setf graph-data (first graph-data))
    (unless legend2 (setf legend2 title))
    (setf graph 
          (tourplot graph-data                        
                    :variable-labels variable-labels 
                    :point-labels point-labels
                    :title title 
                    :location location 
                    :size size
                    :show nil))
    (when graph 
          (send graph :initial-tour))
    (if actcon 
        (enable-container actcon)
        (disable-container))
    (when graph
          (send graph :title (strcat 
             title " [" (if data-object (send data-object :name) "Unnamed Data") "]"))
          (send graph :point-color (iseq (send graph :num-points)) 'blue)
          (send graph :point-label (iseq (send graph :num-points)) point-labels)
          (send (second (send graph :overlays)) :remove-button ':pop)
          (send (second (send graph :overlays)) :remove-button ':zoom)
          (send (second (send graph :overlays)) :remove-button ':box)
          (send graph :switch-add-box)
          (send graph :make-two-plot-menus
                "OrbitPlot"
                :hotspot-items '(help dash link dash 
                                 faster slower cuing axes box dash
                                 show-plots hide-plots close-plots dash
                                 print save copy dash on-top maximize)
                :popup-items   '(showing-labels mouse resize-brush dash 
                                 select-all unselect-all show-all dash 
                                 erase-selection focus-on-selection view-selection dash
                                 color symbol dash selection slicer))
          (send graph :after-new-plot pop-out top-most show size container linkable linked
                data-object content-only)
          (when linkable (send graph :linked linked))
          )
    graph))

